## -*-Tcl-*- (nowrap)
 # ==========================================================================
 #  BibTeX mode - an extension package for Alpha
 # 
 #  FILE: "bibtexEntries.tcl"
 #                                    created: 08/17/1994 {09:12:06 am} 
 #                                last update: 11/07/2001 {10:32:34 AM}
 #                                
 #  Description: 
 #  
 #  Menu item procedures that pertain to entry formatting, validating, and
 #  searching, as opposed to general file or preference operations.
 #  
 #  See the "bibtexMode.tcl" file for license info, credits, etc.
 #  
 # ==========================================================================
 ## 

proc bibtexEntries.tcl {} {}

# load main bib file!
bibtexMenu

namespace eval Bib {}

# ===========================================================================
# 
#  ----  #
# 
#  New Entries, Fields  #
#

# (In Alpha7 the menu might send "Custom Entry" as the $itemName if the menu
# list is long...)
        
proc Bib::entriesProc {menuName itemName} {
    
    Bib::BibModeMenuItem
    
    if {$menuName == "entries" && [getModifiers]} {
        Bib::editEntryFields $itemName
    } elseif {$itemName == "customEntry" || $itemName == "Custom Entry"} {
        Bib::customEntry
    } else {
        Bib::newEntry $itemName
    } 
}

# ===========================================================================
# 
# Create a custom bibliographic entry with user specified fields.
#

proc Bib::customEntry {} {
    
    global BibmodeVars Bib::Fields Bib::OpenEntry Bib::CloseEntry
    
    goto [lindex [Bib::getEntry [getPos]] 1]
    set entryName [prompt "Enter the name of the custom entry:" ""]
    # Choose the first field ...
    set fieldList [listpick -L {author} -p \
      "Pick the first field:" [set Bib::Fields]]
    # ... and remove it from the list ...
    set nextFields [concat  "Complete" [set Bib::Fields]]
    set fieldSpot  [lsearch  $nextFields $fieldList]
    set nextFields [lreplace $nextFields $fieldSpot $fieldSpot]
    # ... and offer the remaining fields.
    while {[set nextField [listpick -L {Complete} -p \
      "Choose another field, or press Complete  :" $nextFields]] != "Complete"} {
        append fieldList " $nextField"
        set fieldSpot  [lsearch  $nextFields $nextField]
        set nextFields [lreplace $nextFields $fieldSpot $fieldSpot]
        dialog::alert "Current fields:  $fieldList"
    }
    # Offer to make it a default menu itemName.
    if {[askyesno "Do you want to make \"$entryName\" a default menu item? \
      \rIt will then be available as an electric completion, too."] == "yes"} {
        # Check to make sure that there are no spaces.
        if {[regsub { } $entryName {} dummy]} {
            status::errorMsg "Cancelled -- no spaces allowed in entry names !!"
        } 
        # Check to make sure that the first word is not Capitalized.
        set first [string tolower [string index $entryName 0]]
        if {$first != [set First [string index $entryName 0]]} {
            set entryName [concat $first[string range $entryName 1 end]]
            dialog::alert "The new entry will be \"$entryName\""
        } 
        # Check to see if $entryName is already defined as a keyword somewhere.
        if {[set type [Bib::checkKeywords $entryName 1]] != 0} {
            dialog::alert "\"$entryName\" is already defined as a $type."
        } else {
            set customEntry [Bib::entryPrefConnect $entryName]
            foreach field $fieldList {
                append fList "$field "
            }
            set BibmodeVars($customEntry) $fList
	    prefs::modified BibmodeVars($customEntry)
            Bib::updatePreferences customEntry
        }
    }
    # Append the fields, all nicely formatted.
    set lines "@${entryName}[set Bib::OpenEntry],\r"
    set length 0
    foreach field $fieldList {
        set fieldLength [string length $field]
        if {$fieldLength > $length} {set length $fieldLength}     
    }
    foreach field $fieldList {
        catch {append lines [Bib::newField $field $length]}
    }
    # Wrap it up.
    append lines "[set Bib::CloseEntry]\r"
    elec::Insertion $lines
}

# ===========================================================================
# 
# Create a new bibliographic entry with its required fields.
#

proc Bib::newEntry {entryName} {
    
    global BibmodeVars Bib::RqdFlds Bib::MyFlds
    global Bib::OpenEntry Bib::CloseEntry Bib::OpenQuote Bib::CloseQuote
    

    # Are we currently in an entry?
    set pos    [getPos]
    if {[catch {Bib::getEntry $pos 1} limits]} {set limits [list $pos $pos]} 
    set pos0   [lindex $limits 0]
    set pos1   [lindex $limits 1]
    if {[pos::compare  $pos > $pos0] && [pos::compare  $pos < $pos1]} {
        goto $pos1
    } else {
        goto [lineStart $pos]
    }
    if {$entryName == "string"} {
        # A special case for strings.
        elec::Insertion "@string[set Bib::OpenEntry] = [set Bib::OpenQuote][set Bib::CloseQuote][set Bib::CloseEntry]\r"
        return
    } 
    # Do we have a selection highlighted?  If so, we'll use that for the
    # default cite-key.
    if {[isSelection]} {
	regsub -all "\t| " [getSelect] {} citeKey
    } else {
	set citeKey ""
    }
    set lines "@${entryName}[set Bib::OpenEntry]${citeKey},\r"
    set customEntryName [Bib::entryPrefConnect $entryName]
    if {[info exists Bib::MyFlds($entryName)] && [llength [set Bib::MyFlds($entryName)]]} {
        # First see if this is a user defined entry with at least one field.
        set fieldList [set Bib::MyFlds($entryName)]
    } elseif {[info exists BibmodeVars($customEntryName)]} {
        # Then check for a customEntryName preference for the entry.
        set fieldList $BibmodeVars($customEntryName)                
    } elseif {[info exists Bib::RqdFlds($entryName)]} {
        # Or the list defined by Bib::RqdFlds().
        set fieldList [set Bib::RqdFlds($entryName)]
    } else {
        # Oh well...  we tried.
        set fieldList {}
    }
    # Append the fields, all nicely formatted.
    set length 0
    foreach field $fieldList {
        set fieldLength [string length $field]
        if {$fieldLength > $length} {set length $fieldLength}     
    }
    foreach field $fieldList {
        catch {append lines [Bib::newField $field $length]}
    }
    # Wrap it up.
    append lines "[set Bib::CloseEntry]"
    # Do we need an extra carriage return at the end?
    set line [getText [getPos] [nextLineStart [getPos]]]
    if {![regexp "^\[\r\n\t \]*$" $line]} {
	append lines "\r"
    } 
    elec::Insertion $lines
}

# ===========================================================================
# 
# Create a new field within the current bibliographic entry.  This includes
# the options for creating a user defined fields, or choosing multiple
# fields.  (In Alpha7 the menu might send "Custom Field" or "Multiple
# Fields" as the $item if the menu list is long...)
#

proc Bib::fieldsProc {menuName itemName} {
    
    Bib::BibModeMenuItem

    global BibmodeVars Bib::Fields Bib::OpenEntry Bib::CloseEntry
    
    if {$menuName == "fields" && [getModifiers]} {
        Bib::editPreference "addFields" "Edit the \"Add Fields\" preference:" 1
        Bib::updatePreferences addFields
        status::msg "The custom fields include \"$BibmodeVars(addFields)\""
    } 
    if {$itemName == "customField" || $itemName == "Custom Field"} {
        # Prompt for name of new field, offer to include it in prefs.
        set newFieldName [prompt "Enter the new field's name:" ""]
        if {[askyesno "Do you want to make \"$newFieldName\" a default menu item?\
          \rIt will then be available as an electric completion, too."] == "yes"} {
            if {[regsub { } $newFieldName {} dummy]} {
                status::errorMsg "Cancelled -- no spaces allowed in field names !!"
            } 
            # Check to make sure that the first word is not Capitalized.
            set first [string tolower [string index $newFieldName 0]]
            if {$first != [set First  [string index $newFieldName 0]]} {
                set newFieldName [concat $first[string range $newFieldName 1 end]]
                dialog::alert "The new entry will be \"$newFieldName\""
            } 
            if {[set type [Bib::checkKeywords $newFieldName 1]] != 0} {
                dialog::alert "\"$newFieldName\" is already defined\
                  in the $type list"
            } else {
                append BibmodeVars(addFields) " $newFieldName"
                set BibmodeVars(addFields) [lsort $BibmodeVars(addFields)]
		prefs::modified BibmodeVars(addFields)
                Bib::updatePreferences addFields
            }
        } 
        set lines [Bib::newField $newFieldName]
    } elseif {$itemName == "multipleFields" || $itemName == "Multiple Fields"} {
        # Prompt for several fields to be inserted at once.
        set flds [listpick -l -L {author year} -p \
          "Pick desired fields:" [set Bib::Fields]]
        if {[llength flds]} {
            set lines {}
            foreach fld $flds {append lines [Bib::newField $fld]}
        } else {
            return
        }
    } else {
        # Else, prepare to insert a field, no questions asked.  Also no
        # length (pad) specified -- entry will just have to be reformatted.
        set lines [Bib::newField $itemName 0] 
    }
    # Where are we?
    set pos  [getPos]
    set pos0 [lineStart  $pos]
    set pos1 [pos::math  $pos0 + [string length $BibmodeVars(indentString)]]
    if {![catch {Bib::getEntry $pos 1} limits]} {
	set pos2 [lindex $limits 0]
	set pos3 [lindex $limits 1]
    } else {
	set pos2 [nextLineStart $pos]
	set pos3 [nextLineStart [nextLineStart $pos]]
    }
    # Where should we go?  Have to handy several different cases here,
    # because we might be building an entry as we go, or adding to an
    # existing entry, or in the middle of a field somewhere ...
    if {[catch {Bib::nextField} pos4]} {
        set pos4 [nextLineStart $pos]
    } 
    if {[pos::compare $pos >= $pos0] && [pos::compare $pos <= $pos1]} {
        # Are we at the start of a line, or at least within the parameter
        # set by the "indentString" pref?  If so, put the new field here.
        goto $pos0

    } elseif {[pos::compare $pos4 > $pos2] && [pos::compare $pos4 < $pos3]} {
	# The 'next' field is within this entry, so put the new field in
	# front of it.
	goto [lineStart $pos4]
    } else {
	# Otherwise, put it at the end the current entry.
	goto [lineStart [pos::math $pos3 - 1]]
    }
    # Do we really need that extra carriage return at the end?
    if {[regexp {^[\r\n\t ]*$} [getText [getPos] [nextLineStart [getPos]]]]} {
        set lines [string trimright $lines]
    } 
    elec::Insertion $lines
}

proc Bib::newField {fieldName {length 0}} {
    
    global BibmodeVars Bib::OpenQuote Bib::CloseQuote Bib::Indent
    global Bib::DefaultFields Bib::UseBrace Bib::FieldDefs Bib::DefFldVal 

    set spc "                   "
    
    # Need braces?
    if {[lsearch -exact [set Bib::DefaultFields] $fieldName] >= 0} {
        set needBraces [set Bib::UseBrace($fieldName)]
    } else {
        set needBraces 1
    }
    # Any default values?
    if {[lsearch -exact [set Bib::FieldDefs] $fieldName] >= 0} {
        set value [set Bib::DefFldVal($fieldName)]
    } else {
        set value ""
    }
    # Were we given a length to use for a pad?
    if {$length} {
        set pad [string range $spc 1 [expr \
          {$length - [string length $fieldName]}]]
    } else {
        set pad ""
    }
    # Return the field to the proc that called this one.
    if {$needBraces} {
        return \
          "[set Bib::Indent]$fieldName =$pad [set Bib::OpenQuote]${value}[set Bib::CloseQuote],\r"
    } else {
        return \
          "[set Bib::Indent]$fieldName =$pad $value,\r"
    }   
}

# ===========================================================================
# 
# Find the next field in the file.  It is up to the calling proc to
# determine if it is in the current entry.
# 

proc Bib::nextField {{pos ""}} {
    
    if {![string length $pos]} {set pos [getPos]}
    set pos [nextLineStart $pos]
    set pat {^[\t ]*[a-zA-Z]+[\t ]*=[\t ]*}
    if {![catch {search -f 1 -r 1 -s $pat $pos} match]} {
	set pos0 [lindex $match 0]
	set lwhite [text::indentString $pos0]
	return [pos::math $pos0 + [string length $lwhite]]
    } else {
	error "No next field found."
    }
}

# ===========================================================================
# 
#  ----  #
# 
#  Navigation, etc  #
#

# ===========================================================================
# 
# Return the bounds of the bibliographic entry surrounding the current
# position.
#

proc Bib::getEntry {pos {quietly 0}} {

    global Bib::TopPat

    if {![catch {search -s -f 0 -r 1 [set Bib::TopPat] $pos} results]} {
	set pos0 [lineStart [lindex $results 0]]
	set pos1 [lindex $results 1]
	if {[catch {matchIt [lookAt [pos::math $pos1 - 1]] $pos1} pos2]} {
	    if {!$quietly} {
		set alert    "There seems to be a badly delimited field in here.\r"
		append alert "Entry or field delimiters might be set incorrectly."
		alertnote $alert
	    } 
	    goto $pos0
	    error "Can't find close brace"
	} else {
	    set pos2 [nextLineStart $pos2]
	}
    } else {
	set pos0 [nextLineStart $pos]
	set pos2 $pos0
    }
    return [list $pos0 $pos2]
}

# ===========================================================================
# 
# Advance to the next bibliographic entry.
#

proc Bib::nextEntry {{quietly 0} {insertTo 0}} {
    
    Bib::BibModeMenuItem
    
    global Bib::TopPat
    
    set posBeg  [getPos]
    set posEnd  [selEnd]
    if {![catch {Bib::getEntry $posEnd 1} limits]} {
	set pos0 [lindex $limits 1]
    } else {
	set pos0 $posBeg
    }
    set pos1    $pos0
    set nextPos [lineStart $pos1]

    while {![catch {search -f 1 -r 1 -s [set Bib::TopPat] $pos1} pos]} {
        regexp [set Bib::TopPat] [eval getText $pos] match type
        if {$type != "string"} {
            set nextPos [lindex $pos 0]
            break
        } else {
            set pos1 [nextLineStart [lindex $pos 1]]
        }
    }
    if {![catch {Bib::getEntry $nextPos 1} limits]} {
	set pos2 [lindex $limits 1]
    } else {
	set pos2 $posEnd
    }
    if {[isSelection]} {
        if {[pos::compare $posEnd < $pos0]} {
            select $posBeg $pos0
        } else {
            select $posBeg $pos2
        }
	status::msg ""
    } elseif {[pos::compare $posEnd == $pos2]} {
        status::msg "No further entries in this file."
	return ""
    } else {
        goto $nextPos
        set nextEntry [Bib::copyCiteKey 0]
        if {$quietly} {return [list $nextPos $nextEntry]}
        status::msg "$nextEntry"
        if {$insertTo == 1} {insertToTop} elseif {$insertTo == 2} {centerRedraw}
    }
}

# ===========================================================================
# 
# Go back to the previous bibliographic entry.
#

proc Bib::prevEntry {{quietly 0} {insertTo 0}} {
    
    Bib::BibModeMenuItem
    
    global Bib::TopPat
    
    set posBeg [getPos]
    set posEnd [selEnd]
    if {![catch {Bib::getEntry $posBeg 1} limits]} {
	set pos0 [lindex $limits 0]
	set pos1 [lindex $limits 1]
    } else {
	set pos0 $posBeg
	set pos1 $posEnd
    }
    set prevPos $pos0
    if {[pos::compare $pos0 > [minPos]] && [pos::compare $pos0 >= $posBeg]} {
        set pos0 [pos::math $pos0 - 1]
        while {![catch {search -f 0 -r 1 -s [set Bib::TopPat] $pos0} pos]} {
            regexp [set Bib::TopPat] [eval getText $pos] match type
            if {$type != "string"} {
                set prevPos [lindex $pos 0]
                break
            } else {
                set pos0 [lineStart [lindex $pos 0]]
                if {[pos::compare $pos0 == [minPos]]} {break}
                set pos0 [pos::math $pos0 - 1]
            }
        }
    }
    if {[isSelection]} {
	select $prevPos $posEnd
	status::msg ""
    } elseif {[pos::compare $posBeg == $prevPos]} {
        status::msg "No further entries in this file."
        return ""
    } else {
        goto $prevPos
        set prevEntry [Bib::copyCiteKey 0]
        if {$quietly} {return [list $prevPos $prevEntry]}
	status::msg "$prevEntry"
        if {$insertTo == 1} {insertToTop} elseif {$insertTo == 2} {centerRedraw}
        return $prevEntry 
    }
}

# ===========================================================================
# 
# Use 1 and 3 on the keypad to navigate.
#

proc Bib::searchFunc {direction} {
    
    if {$direction} {Bib::nextEntry 0 2} else {Bib::prevEntry 0 2}
}


# ===========================================================================
# 
# Select (highlight) the current bibliographic entry.
#

proc Bib::selectEntry {} {

    Bib::BibModeMenuItem
    
    set pos [Bib::getEntry [getPos]]
    select [lindex $pos 0] [lindex $pos 1]
}

# ===========================================================================
# 
# Put the cite-key of the current entry on the clipboard.
# 

proc Bib::copyCiteKey {{copy 1}} {
    
    Bib::BibModeMenuItem
    
    set where [getPos]
    global Bib::TopPat2
    set limits [Bib::getEntry $where]
    set top    [lindex $limits 0]
    set bottom [lindex $limits 1]
    if {[regexp -indices [set Bib::TopPat2] [getText $top $bottom] allofit type citeKey]} {
        set start [pos::math $top + [lindex $citeKey 0]]
        set end   [pos::math $top + [expr [lindex $citeKey 1] + 1]]
        select $start $end
        set citeKey [getSelect]
        if {$copy} {
            copy
            status::msg "Copied \"$citeKey\" to clipboard."
        }
        goto $where
        return $citeKey
    } else {
        status::msg "Couldn't find cite-key."
        return  "Couldn't find cite-key."
    }
}

# ===========================================================================
# 
# Adapted from 'beginningOfLineSmart' and 'Shel::Bol'.
# 
# If we're in a line that starts a field, move the cursor to the beginning
# of the text within that field, otherwise move the cursor in front of the
# first word.  Pressing 'control-a' again will move the cursor to the
# beginning of the line, and again will toggle these two positions.
# 

proc Bib::beginningOfLineSmart {} {
    
    set pos0 [getPos]
    set pos1 [lineStart $pos0]
    set lim  [nextLineStart $pos1]
    set pat {[a-zA-Z]+[\t ]*=[\t \{"]*}
    if {![catch {search -f 1 -r 1 -s -l $lim $pat $pos1} fieldStart]} {
	set pos2 [lindex $fieldStart 1]
    } else {
	set pos2 [text::firstNonWsLinePos $pos0]
    }
    if {[pos::compare $pos0 == $pos2]} {
	beginningOfLine
    } else {
	goto $pos2
    }
}

# ===========================================================================
# 
#  ----  #
# 
#  Formatting, Validating  #
#
        
# ===========================================================================
# 
# Format Entry
# 
# Parse the entry around position "pos" and rewrite it to the original
# buffer in a canonical format
#

proc Bib::formatEntry {{fromClick 0}} {
    
    Bib::BibModeMenuItem
    
    global BibmodeVars Bib::UseBrace Bib::OpenQuote Bib::CloseQuote Bib::Abbrevs
    global Bib::OpenEntry Bib::CloseEntry Bib::Strings
    
    # Make sure that we're not in a selection.
    goto [getPos]

    set limits [Bib::getEntry [getPos]]
    set top    [lindex $limits 0]
    set bottom [lindex $limits 1]
    
    if {[pos::compare [getPos] == [maxPos]]} {
        status::msg "No further entries in this file to format."
        return
    }
    # Convert tabs to spaces in the entry, else it can get funky.
    select $top $bottom 
    tabsToSpaces
    goto $top
    
    # Now we need to find the bottom again.
    set limits [Bib::getEntry [getPos]]
    set top    [lindex $limits 0]
    set bottom [lindex $limits 1]
    
    status::msg "Formatting "
    set Bib::Strings [Bib::listStrings]
    
    if {[catch {Bib::bibFormatEntry $top} result]} {
        status::errorMsg "Couldn't format this entry for some reason"
    }
    if {$result != [getText $top $bottom]} {replaceText $top $bottom $result}
    goto $top
    insertToTop
    set nextEntry [lindex [Bib::nextEntry 1] 1]
    if {$nextEntry == "Couldn't find cite-key."} {
        status::msg "No further entries in this file to format."
    } else {
	catch {goto [Bib::nextField]}
        if {$fromClick} {
            set do "Command double click on the next set of fields"
        } else {
            set do "Press \"control-shift-L\""
        } 
        status::msg "$do to format  $nextEntry"
    }
}

# ===========================================================================
# 
# Format All Entries
# 
# Parse all entries in the current buffer and rewrite it to the original
# buffer in a canonical format.  If any entries cannot be formatted, we'll
# put that info in a little Results window.
#

proc Bib::formatAllEntries {} {

    Bib::BibModeMenuItem
    
    global BibmodeVars Bib::UseBrace Bib::OpenQuote Bib::CloseQuote Bib::Abbrevs
    global Bib::OpenEntry Bib::CloseEntry Bib::Strings

    # Make sure that we're not in a selection.
    goto [getPos]
    # This little dance handles the case that the first entry starts on the
    # first line.
    set hit [Bib::getEntry [getPos]]
    if {[pos::compare [lindex $hit 0] == [lindex $hit 1]]} {
        Bib::nextEntry 1
        set hit [Bib::getEntry [getPos]]
    }
    # Set up the variables for the report.
    set count        0  
    set errorCount   0
    set formatResult ""
    # Format all of the entries.
    status::msg "Formatting "
    # Create the list of all strings and abbrevs.  If there are many of them,
    # this will go pretty slow.
    set Bib::Strings [Bib::listStrings]
    
    while {[pos::compare [getPos] < [lindex $hit 1]]} {
        # Convert tabs to spaces for this entry, else things can get funky.
        Bib::selectEntry
        status::msg "Formatting: [incr count]"
        tabsToSpaces
        status::msg "Formatting: $count"
        set hit [Bib::getEntry [lindex $hit 0]]
        set top [lindex $hit 0] 
        set bottom [lindex $hit 1]
        # Now try to format the entry.
        if {![catch {Bib::bibFormatEntry $top} result]} {
            # bibFormatEntry didn't fail, so replace the text if necessary.
            set oldEntry [getText $top $bottom]
            if {$result != $oldEntry} {
                deleteText $top $bottom 
                insertText $result
            } 
        } else {
            # There was some sort of error ...
            incr errorCount
            append formatResult "\r[format {%-17s} "line [lindex [posToRowCol $top] 0]"]"
            catch {append formatResult ", cite-key \"[lindex [lindex [Bib::getFields $top] 1] 1]\""}
        }
        goto $top
        # Go to the next entry.
        insertToTop
        status::msg "Formatting: $count"
        Bib::nextEntry 1
        set hit [Bib::getEntry [getPos]]
        # And a little insurance ...
        if {[pos::compare [getPos] == [maxPos]]} {break}
    }
    if {!$errorCount} {
        status::msg "$count entries formatted.  No errors detected."
    } else {
        # We had errors, so we'll return them in a little window.
        status::msg "$count entries formatted.  Errors detected "
        set t    "% -*-Bib-*- (validation)\r"
        append t "\r  Formatting Results for \"[win::CurrentTail]\"\r\r"
        append t "  Note: Command double-click on any cite-key or line-number\r"
        append t "        to return to its original entry.  If there is no\r"
        append t "        cite-key listed, that is certainly one problem ...\r\r"
        append t "___________________________________________________________\r\r"
        append t "    Formatted Entries:  [format {%4d} [expr $count - $errorCount]]\r\r"
        append t "  Unformatted Entries:  [format {%4d} $errorCount]\r"
        append t "___________________________________________________________\r\r"
        append t "  line numbers:  cite-keys:\r"
        append t "  -------------  ----------\r"
        append t $formatResult
        new -n "* Formatting Results *" -m "Bib" -text $t
        winReadOnly
        shrinkHigh
    }
}

# ===========================================================================
# 
# Parse the entry around position "pos" and rewrite it in a canonical
# format.  The formatted entry is returned.
#

proc Bib::bibFormatEntry {pos} {
    
    global BibmodeVars Bib::RqdFlds Bib::OptFlds Bib::UseBrace Bib::EntryNameConnect 
    global Bib::Strings Bib::Abbrevs
    global Bib::OpenQuote Bib::CloseQuote Bib::OpenEntry Bib::CloseEntry Bib::Indent
    
    set strings [concat [set Bib::Strings] [set Bib::Abbrevs]]

    if {[catch {Bib::getFields $pos} fieldLists]} {
        error "bibFormatEntry: \"Bib::getFields\" couldn't find any"
    }
    set names   [lindex  $fieldLists 0]
    set values  [lindex  $fieldLists 1]
    
    # Get the name of the entry, make it lower case, and then (if possible)
    # connect it to the default entryName.
    set entrytype [string tolower [lindex $values 0]]
    set entryType $entrytype
    catch {set entryType [set Bib::EntryNameConnect($entrytype)]}
    
    # Don't process @string entries.
    if {$entryType == "string"} {
        set limits [Bib::getEntry $pos]
        set lines  [getText [lindex $limits 0] [lindex $limits 1]]
        return $lines
    }
    # Get the cite-key.  If it doesn't exists, this proc already failed
    # in Bib::getFields.
    set citeKey [lindex $values 1]
    # Find length of longest field name.
    set length 0
    foreach field $names {
        set fieldLength [string length $field]
        if {$fieldLength > $length}          {set length $fieldLength}
        if {![info exists Bib::UseBrace($field)]} {set Bib::UseBrace($field) 1}
    }
    
    # Format first line
    set lines "@${entryType}[set Bib::OpenEntry]${citeKey},\r"
    if {![info exists Bib::RqdFlds($entryType)]} {set entryType "misc"} 
    # Format each field on a separate line.  We start with ifld 2 because
    # 0 is the type, and 1 is the citeKey.
    set spc "                           "
    for {set fldIndex 2} {$fldIndex < [llength $names]} {incr fldIndex} { 
        set field [lindex $names  $fldIndex]
        set value [lindex $values $fldIndex]
        # Now we take care of the case that all of 
        # 
        # (1) "zapEmptyFields" is on,
        # (2) there is an empty field present, and
        # (3) "entryType" is not a default entry.
        # 
        # We can do this since entryType is already set into the $lines. 
        # The result will be to zap all empty fields, since none will be
        # required.  If we don't do this, the proc fails ...
        set test1 $BibmodeVars(zapEmptyFields)
        set test2 [lcontains Bib::RqdFlds($entryType) $field]
        if {$value != "" || !$test1 || $test2} {
            set pad [expr $length - [string length $field]]
            if {$BibmodeVars(alignEquals)} {
                set pref "[set Bib::Indent]$field[string range $spc 1 $pad] ="
            } else {
                set pref "[set Bib::Indent]$field =[string range $spc 1 $pad]"
            }
            set ind [string range $spc 1 [string length $pref]]
            # Delimit field, if appropriate
            set test1   [is::UnsignedInteger $value]
            set test2   [regexp {[^\\]\#} $value] 
            set noBrace [expr ([set Bib::UseBrace($field)] == 0 && $test1) || $test2]
            if {$noBrace == 0 && [string first " " $value] < 0} {
                set noBrace [expr [lsearch $strings [string tolower $value]] >= 0]
            }
            if {$noBrace != 0} {
                set value "$value,"
            } else {
                set value "[set Bib::OpenQuote]${value}[set Bib::CloseQuote],"
            }
            set     pieces [split $value "\r"]
            append  lines  "$pref [lindex $pieces 0]\r"
            foreach piece  [lrange $pieces 1 end] {append lines "$ind  $piece\r"}
        }
    }
    append lines "[set Bib::CloseEntry]\r"
    return $lines
}

# ===========================================================================
# 
# Return a list of all @string s
#

proc Bib::listStrings {} {
    
    global Bib::TopPat1 Bib::Strings

    set searchPattern {^[ ]*(@string)}
    set countStrings 1
    set pos [minPos]
    # We'll see if we can avoid re-creating the list of strings.  We set
    # the count to 1 so that if there are not in fact any strings we know
    # that, too.
    while {![catch {search -f 1 -r 1 -m 0 -i 1 -s $searchPattern $pos} match]} {
        set pos [nextLineStart [lindex $match 0]]
        incr countStrings
    }
    set BibStringLength 0
    catch {set BibStringLength [expr [llength [set Bib::Strings]] + 1]}
    if {$BibStringLength == $countStrings && $BibStringLength != 0} {
        # We know that this proc was called once before.  We'll assume that
        # the actual strings haven't changed if the length is the same.
        status::msg ""
        return [set Bib::Strings]
    } else {
        status::msg "scanning for @strings"
        set matches [Bib::findEntries {^[ ]*@string *[\{\(]} 0]
        foreach hit $matches {
            set top [lindex $hit 2] 
            set bottom [lindex $hit 3]
            set entry [getText $top $bottom]
            regsub -all "\[\n\r\]+" $entry { } entry
            regsub -all "\[     \]\[    \]+" $entry { } entry
            regsub {[,  ]*[\)\}][   ]*$} $entry { } entry
            regexp [set Bib::TopPat1] $entry allofit citeKey
            set citeKey [string tolower $citeKey]
            if {[catch {incr strings($citeKey)} num]} {
                set strings($citeKey) 1
            }
        }
        if {[catch {lsort [array names strings]} res]} {set res {}}
        status::msg ""
        return $res
    }
}

# ===========================================================================
# 
# Return a list containing the data for the current entry, indexed by the
# parameter names, e.g., "author", "year", etc.  Index names for the entry
# type and cite-key are "type" and "citeKey".
#

proc Bib::getFields {pos} {

    global Bib::TopPat Bib::TopPat1 Bib::TopPat2 Bib::TopPat3
    
    set fldPat {[   ]*([a-zA-Z]+)[  ]*=[    ]*}
    
    set limits [Bib::getEntry $pos]
    set top [lindex $limits 0]
    set bottom [lindex $limits 1]
    
    set entry [getText $top $bottom]
    regsub -all "\[\n\r\]+"             $entry { } entry
    regsub -all "\[\t \]\[\t \]+"       $entry { } entry
    regsub "\[,\t \]*\[\)\}\]\[\t \]*$" $entry { } entry
    
    if {[regexp -indices [set Bib::TopPat2] $entry match theType theKey ]} {
        set key [string range $entry [lindex $theKey 0] [lindex $theKey 1]]
        set theRest [expr 1 + [lindex $match 1]]
    } elseif {[regexp -indices [set Bib::TopPat3] $entry match theType aField]} {
        set key {}
        set theRest [lindex $aField 0]
    } else {
        error "Invalid entry"
    }
    lappend names type
    set type [string tolower [string range $entry [lindex $theType 0] [lindex $theType 1]]]
    lappend data $type
    
    lappend names citeKey
    lappend data $key
    
    set entry  ",[string range $entry $theRest end]"
    set fldPat ",\[\t \]*(\[^ =,\]+)\[\t \]*=\[\t \]*"
    set name   ""
    while {[regexp -indices $fldPat $entry match sub1]} {
        set nextName [string range $entry [lindex $sub1 0] [lindex $sub1 1]]
        lappend names [string tolower $nextName]
        if {$name != ""} { 
            set prevData [string range $entry 0 [expr [lindex $match 0]-1]]
            lappend data [breakIntoLines [Bib::bibFieldData $prevData]]
        }   
        set name $nextName
        set entry [string range $entry [expr [lindex $match 1]+1] end]
    }
    
    lappend data [breakIntoLines [Bib::bibFieldData $entry]]
    
    return [list $names $data]
}

proc Bib::bibFieldData {text} {

    set text [string trim $text]
    # Preserve an entire {{Odd Name}} or "{Odd Name}" or {"Odd Name"} field.
    if {[regexp {(^(\{|\")\{.+\}(\}|\")$)} $text]} {
        regsub {^(\{|\")} $text {} text
        regsub {(\}|\")$} $text {} text
        return $text
    } 

    set text  [string trim $text {   ,#}]
    set text1 [string trim $text {\{\}\" }]
    
    if {[string match {*[\{\}\"]*} $text1]} {
        set words [parseWords $text]
        if {[llength $words]==1} {
            regsub {^[\{\"\']} $text {} text
            regsub {[\}\"\']$} $text {} text
        }
    } else {
        set text $text1         
    }
    return $text
}

# ===========================================================================
# 
# Validate Entry
# 
# Check the fields of the current entry against the Bib::RqdFlds() and
# Bib::ValidFlds() arrays, and report any irregularities.  The argument
# "quietly" is set to 1 when this is called by "Validate All Entries."
#
# When called from Bib::validateAllEntries, the result is a list:
# 
# (0) the cite-key
# (1) a (possibly empty) list of missing fields, set to "-1" if the
#     entry was not recognized by Bib::RqdFlds()
# (2) a (possibly empty) list of "extra" fields.
# 

proc Bib::validateEntry {{quietly 0}} {
    
    Bib::BibModeMenuItem
    
    global BibmodeVars Bib::EntryNameConnect Bib::RqdFlds Bib::ValidFlds
    
    # Make sure that we're not in a selection.
    goto [getPos]
    # Get the field array.
    if {[catch {Bib::getFields [getPos]} fieldLists]} {
        status::errorMsg "Error: could not validate this entry"
    }
    # set fieldLists [Bib::getFields [getPos]]
    # Get the cite-key.
    set citeKey   [lindex [lindex $fieldLists 1] 1]
    # Get entry type -- this will be lower case, which we then connect.
    set entrytype [lindex [lindex $fieldLists 1] 0]
    set entrytype [string tolower $entrytype]
    set entryType "dummy"
    catch {set entryType [set Bib::EntryNameConnect($entrytype)]}
    # If this is a "string", it must have been at the top of the file.
    if {$entryType == "string"} {
        Bib::nextEntry 1
        return [list "string!!!"]
    } 
    # get field list, then remove "type" and "citeKey", and flush braces.
    set fieldList [lindex $fieldLists 0]
    set fieldList [lreplace $fieldList 0 1]
    regsub -all "\{" $fieldList "" fieldList
    regsub -all "\}" $fieldList "" fieldList
    # Now check to see if we have any info on valid fields.  Note that we
    # return an "error" even if this is a user defined entry.
    if {![info exists Bib::ValidFlds($entryType)]} {
        Bib::nextEntry 1
        if {!$quietly} {
            status::msg  "Sorry, no field information available\
              for the entry \"$entrytype\"."
        } 
        return [list $citeKey "-1 $entrytype" ""]
    } 
    # Now we're ready to validate.  First check for required fields:
    set missingFields ""
    foreach field [set Bib::RqdFlds($entryType)] {
        if {[lsearch $fieldList $field] == "-1"} {
            append missingFields "$field "
        } 
    }
    # Then check for extra fields:
    set nSFields ""
    if {!$BibmodeVars(ignoreExtraFields)} {
        foreach field $fieldList {
            if {[lsearch [set Bib::ValidFlds($entryType)] $field] == "-1"} {
                append nSFields "$field "               
            } 
        }
    } 
    # Return the results:
    set result "$citeKey -- "
    if {[llength $missingFields]} {
        append result "missing fields: $missingFields"
    } else {
        append result "no missing fields "
    } 
    if {[llength $nSFields]} {
        append result "; extra fields: $nSFields"
    } 
    insertToTop
    Bib::nextEntry 1
    if {$quietly} {
        return [list $citeKey $missingFields $nSFields]
    } else {
        status::msg $result
    }
}

# ===========================================================================
# 
# Validate All Entries
# 
# Check the fields of each entry in the current window against the Bib::RqdFlds()
# and Bib::OptFlds() arrays, and report any irregularities in a new window.  Also
# checks for duplicate cite-keys.  Note that this only validates from the
# current entry on down, to be consistent with "Format All Entries".
#


proc Bib::validateAllEntries {} {
    
    Bib::BibModeMenuItem
    
    global BibmodeVars
    
    # Make sure that we're not in a selection.
    goto [getPos]
    # This little dance handles the case that the first entry starts on the
    # first line.
    set hit [Bib::getEntry [getPos]]
    if {[pos::compare [lindex $hit 0] == [lindex $hit 1]]} {
        Bib::nextEntry 1
        set hit [Bib::getEntry [getPos]]
    }
    # Set up the variables for the report.
    set results        ""
    set citeKeyList    ""
    set count          0 
    set missingCount   0
    set nSFieldCount   0
    set duplicateCount 0
    set unknownCount   0
    # Now validate all of the entries.
    while {[pos::compare [getPos] < [lindex $hit 1]]} {
        set validateList [Bib::validateEntry 1]
        set report 0
        set citeKey [lindex $validateList 0]
        # (Special quirk if a string is at the top of the file.)
        if {$citeKey == "string!!!"} {
            set hit [Bib::getEntry [getPos]]
            continue
        } 
        status::msg "Validating: [incr count]"
          # Check for duplicate cite keys.
        set result  "[format {%-25s} "$citeKey  "]"
        if {[lsearch $citeKeyList $citeKey] != "-1"} {
            append result "\r  Warning: \"$citeKey\" is a duplicate cite-key."
            set report 1
            incr duplicateCount
        } 
        lappend citeKeyList $citeKey
        # Any missing fields?  If the list is empty, we do nothing.
        set missingFields [lindex $validateList 1]
        if {[lindex $missingFields 0] == "-1"} {
            # This was an unrecognized entry.
            append result "\r  Warning: no field information available\
              for the entry \"[lindex $missingFields 1]\"."
            set report 1
            incr unknownCount
        } elseif {$missingFields != ""} {
            # Missing fields for the recognized entry.
            append result "$missingFields"
            set report 1
            incr missingCount
        } 
        # Any "extra" fields?  If the list is empty, we do nothing.
        set nSFields  [lindex $validateList 2]
        if {$nSFields != ""} {
            append result "\r         extra fields:   $nSFields"
            set report 1
            incr nSFieldCount
        } 
        # Did we found out anything?
        if {$report} {append results "$result\r"} 
        # We're already at the start of the next entry.
        set hit [Bib::getEntry [getPos]]
        # And a little insurance ...
        if {[pos::compare [getPos] == [maxPos]]} {break}
    }
    # If no irregularities were found, just return a happy message. 
    # Otherwise, generate a "Validation Results" report.
    if {$results == ""} {
        status::msg "$count entries validated -- no irregularities found."
        return
    } 
    # Do we want to append the results to a current "* Format Results *"
    # window or not?
    set appendResults   0
    set validateWindows ""
    foreach window [winNames] {
        # We'll use this in a moment.
        if {[regexp "\\* Validation Results \\*" [win::StripCount $window]]} {
            lappend validateWindows $window
        } 
    }
    if {[llength $validateWindows]} {
        # Find out if we should append to a current search window.
        if {[askyesno "Would you like to append the validation results\
          \rto the current \"Validation Results\" window?"] == "yes"} {
            # Now we need to see if there's more than one.
            if {[llength $validateWindows] != 1} {
                set appendResults [listpick -p \
                  "Please choose a window:" $validateWindows]
            } else {
                set appendResults [lindex $validateWindows 0]
            } 
        } 
    } 
    # Generate the report header.
    append t "\r  Validation Results for \"[win::CurrentTail]\"\r\r"
    append t "  Note: Cite-keys followed by an empty string do not have missing fields.\r"
    append t "        Command double-click on any cite-key to return to its entry.\r"
    append t "________________________________________________________________________\r\r"
    # Add validation summary results.
    append t "     Validated Entries:  [format {%4d} $count]\r\r"
    append t "        Missing Fields:  [format {%4d} $missingCount]\r"
    if {!$BibmodeVars(ignoreExtraFields)} {
        append t "          Extra Fields:  [format {%4d} $nSFieldCount]\r"
    } 
    append t "   Duplicate Cite-keys:  [format {%4d} $duplicateCount]\r"
    append t "  Unrecognized Entries:  [format {%4d} $unknownCount]\r"
    append t "________________________________________________________________________\r\r"
    append t "  cite-keys:             missing fields:\r"
    append t "  ----------             ---------------\r\r"
    append t "$results\r"
    append t "________________________________________________________________________\r"
    append t "________________________________________________________________________\r"
    # Either create a new window, or append to an existing one.
    placeBookmark
    if {$appendResults == 0} {
        set    t1 "% -*-Bib-*- (validation)\r"
        append t1 $t
        new -n "* Validation Results *" -m "Bib" -text $t1
        set pos [minPos]
    } else {
        bringToFront $appendResults
        setWinInfo read-only 0
        goto [maxPos]
        set pos [getPos]
        insertText $t
    }
    # Color warnings, errors, etc. red.
    set searchText {cite-keys:|missing fields:|Warning:|extra fields:}
    set pos2 [minPos]
    while {![catch {search -f 1 -i 0 -s -r 1 $searchText $pos2} match]} {
        text::color [lindex $match 0] [lindex $match 1] 5
        set pos2 [lindex $match 1]
    } 
    refresh
    winReadOnly
    # Now we'll mark just those entries that truly have missing fields. 
    # Handy if "ignoreExtraFields" is turned off and there are a lot of
    # entries with extra fields.
    removeAllMarks
    set pos3 [minPos]
    set searchText {^([-a-zA-Z0-9_:/\.])+([\t ]+[a-zA-Z])}
    while {![catch {search -f 1 -r 1 -m 0 -i 0 -s $searchText $pos3} match]} {
        set start [lindex $match 0]
        set end   [nextLineStart $start]
        set t     [getText $start $end]
        regexp {[a-zA-Z0-9]+[-a-zA-Z0-9_:/\.]} $t citeKeyMark
        setNamedMark $citeKeyMark $start $start $start
        set pos3 $end
    }
    goto $pos ; insertToTop
    status::msg "Entries with missing fields are listed in the marks menu."
}

# ===========================================================================
# 
#  ----  #
# 
#  Searching  #
# 

# ===========================================================================
# 
# Find all entries that match a given regular expression and copy them to a
# new buffer.
#

proc Bib::searchEntries {{pattern ""} {wCT ""}} {
    
    Bib::BibModeMenuItem
    
    if {$pattern == ""} {
        # Set the text / regexp to search.
        set pattern [prompt "Enter alpha-numeric text or\
          a Regular Expression to search:" ""]
        if {![string length $pattern]} {
            status::errorMsg "Cancelled -- nothing was entered."
        }
    } 
    set reg ^.*$pattern.*$
    # Find any matches.
    set matches [Bib::findEntries $pattern]
    if {[llength $matches] > 0} {
        # "wCT" means that this was called from a "Search All Bib Files"
        # search, and that we want to either report that there were matches 
        # (if "wCT" == "-1") or append the results to $wCT.
        if {$wCT == "-1"} {return 1} 
        Bib::writeEntries $matches 0 [minPos] -1 0 $wCT "all fields" $pattern [llength $matches]
    } else {
        status::msg "No matching entries were found"
        return 0
    }
}

# ===========================================================================
# 
# Find all fields in which the indicated text matches a given regular
# expression and copy them to a new buffer.
#

proc Bib::searchFields {{field ""} {pattern ""} {wCT ""}} {
    
    Bib::BibModeMenuItem
    
    global Bib::Fields

    if {$field == "" || $pattern == ""} {
        # Set the field value to search.
        set fieldList [lsort [concat {"all fields"} "citeKey" [set Bib::Fields]]]
        if {[catch {eval prompt {{Select field type to search:}} \
          "author" {Fields} $fieldList} field]} return
        if {[lsearch $fieldList $field] == "-1"} {
            if {![string length $field]} {
                status::errorMsg "Cancelled -- no field was entered."
            } else {
                status::errorMsg "Cancelled -- \"$field\" is not a valid field."
            }
        } 
        # Set the text / regexp to search.
        set pattern [prompt "Enter alpha-numeric text or\
          a Regular Expression to search:" ""]
        if {![string length $pattern]} {
            status::errorMsg "Cancelled -- nothing was entered."
        }
    } 
    if {$field == "all fields"} {
        # We'll redirect this.
        Bib::searchEntries $pattern $wCT
        return
    } 
    set reg ^.*$pattern.*$
    # Find any matches.
    set matches [Bib::findEntries $pattern]
    if {[llength $matches] == 0} {
        status::msg "No matches were found"
        return "No matches were found"
    }
    set vals {}
    foreach hit $matches {
        set pos [lindex $hit 1]
        set top [lindex $hit 2] 
        set bottom [lindex $hit 3]
        while {[set failure [expr {[Bib::getFldName $pos $top] != $field}]]  && 
        ![catch {search -f 1 -r 1 -i 1 -m 0 -l $bottom -s -- $pattern $pos} match]} {
            set pos [lindex $match 1]
        }
        if {!$failure} { lappend vals [list $top $bottom] }
    }
    if {$wCT == "-2"} {
        # wCT == "-2" means that this was called from control-command
        # double click on a citeKey to check for duplicate citeKeys in
        # the current file.
        if {[llength $vals] == 1 && $field == "citeKey"} {
            status::msg "No duplicate cite-keys for \"$pattern\" were found."
            return 0
        } else {
            set wCT ""
        }
    }
    if {[llength $vals] > 0} {
        # "wCT" means that this was called from a "Search All Bib Files"
        # search, and that we want to either report that there were matches 
        # (if "wCT" == "-1") or append the results to $wCT.
        if {$wCT == "-1"} {
            return 1
        } 
        Bib::writeEntries $vals 0 [minPos] -1 0 $wCT $field $pattern [llength $vals]
    } else {
        status::msg "No matches were found."
        return 0
    }
}

# ===========================================================================
# 
# Search All Bib Files
# 
# Prompt the user for a citation to search for in the list returned by
# "Bib::listAllBibliographies".  Also called by "Bib::noEntryExists".  Can
# also handle regular expressions -- if citations are found in multiple
# files, the user is given the option to view them all in a browser window.
# 
# Results of the search are sent (with the cite-key) to Bib::searchFields. 
# 

proc Bib::searchAllBibFiles {{pattern ""} {field ""} {bibfiles ""} {quietly 0}} {
    
    global Bib::Fields Bib::TopPat

    if {$field == ""} {
        set fieldList [lsort [concat {"all fields"} "citeKey" [set Bib::Fields]]]
        if {[catch {eval prompt {{Select field type to search:}} \
          {"all fields"} {Fields} $fieldList} field]} return
        if {[lsearch $fieldList $field] == "-1"} {
            if {![string length $field]} {
                status::errorMsg "Cancelled -- no field was entered."
            } else {
		status::errorMsg "Cancelled -- \"$field\" is not a valid field."
            }
        } 
    } 
    if {$pattern == ""} {
        set pattern [prompt "Enter alpha-numeric text, \
          or a Regular Expression:" ""]
    } 
    if {[set $pattern [string trim $pattern]] == ""} {
	status::errorMsg "Cancelled -- nothing was entered."
    }
    if {$bibfiles == ""} {
        # Get the list of all bibliographies.  List will be full pathnames.
        set bibfiles [Bib::listAllBibliographies]
    } 
    if {![llength $bibfiles]} {
        dialog::alert "Cancelled -- there are no bibliography files to search.\
          Perhaps the Bib mode \"Use ... Path\" preferences need to be checked."
        error "There are no bibliography files to search."
    }
    # Now do a grep search in all of these files.  If the field is
    # "citeKey", everything is easy.  Otherwise we're going to get a lot of
    # false positives.
    if {$field == "citeKey"} {
        set searchPat "[set Bib::TopPat]\[\t \]*.*($pattern).*,"
    } else {
        set searchPat "^.*($pattern).*$"
    } 
    set biblist ""
    set results  ""
    foreach f $bibfiles {
        if {[set result [grep $searchPat $f]] != ""} {
            lappend biblist $f
            append  results $result
        }
    }
    if {![llength $biblist]} {
        set biblist "-1"
    } 
    status::msg ""
    # Search is over.  Now what to do?
    if {$quietly} {
        return [list $pattern $field $biblist $results]
    } 
    # Otherwise, pass the information along and continue ...
    Bib::reportAllBibFiles $pattern $field $biblist $results
}

# ===========================================================================
# 
# Quick Find Citation
# 
# Prompt the user in the status bar window for citekeys contained in the
# bibIndex file, creating one if necessary.
# 
# Note: the "nomatchiserror" flag allows a match to immediately jump to the
# entry for some reason ...
# 
# The support procs for "Quick Find Citation" are contained in the
# "bibtexData.tcl" file.
#

proc Bib::quickFindCitation {} {
    
    global BibmodeVars PREFS
    
    if {![file exists [file join $PREFS bibIndex]]} {
        Bib::rebuildIndex
        status::msg "Index has been rebuilt -- try again !!"
        return
    } 
    Bib::GotoEntry [prompt::statusLineComplete          \
      "Citation" Bib::completionsForEntry               \
      -nomatchiserror                                   \
      -preeval  {source [file join $PREFS bibIndex]}    \
      -posteval {unset bibIndex}]
}

# ===========================================================================
# 
# Support for the search procs
# 
# The support procs for "Quick Find Citation" are contained in the
# "bibtexData.tcl" file.
#

# ===========================================================================
# 
# Search for all entries matching a given regular expression.  The results
# are returned in a list, each element of which is a list of four integers:
# the beginning and end of the matching entry and the beginning and end of
# the matching string.  Adapted from "matchingLines" in "misc.tcl".
#

proc Bib::findEntries {reg {casesen 1}} {
    if {![string length $reg]} return
    
    set pos [minPos]   
    set result {}                             
    while {![catch {search -f 1 -r 1 -m 0 -i $casesen -s $reg $pos} match]} {
        set entry [Bib::getEntry [lindex $match 0]]
        lappend result [concat $match $entry]
        set pos [lindex $entry 1]
    }
    return $result
}

# ===========================================================================
# 
# Take a list of lists that point to selected entries and copy these into a
# new window.  The beginning and ending positions for each entry must be
# the last two items in each sublist.  The rest of the sublists are
# ignored.  It is assumed that each sublist has the same number of items.
# 
# The optional argument "sort" indicates that this was sent by a
# "sort file" call, which can allow overwriting of the buffer. 
# The optional argument "wCT" is the search window to append to.
# 

proc Bib::writeEntries {pos nonDes {beg ""} {end {-1}} {sort 0} {wCT ""} {field ""} {searchPat ""} {entryLength ""}} {
    
    global BibmodeVars
    if {$beg == ""} {set beg [minPos]}
    if {$end < 0}   {set end [maxPos]}
    # pos is "entry position" ...
    set llen [expr [llength [lindex $pos 0]] - 1]
    set llen1 [expr {$llen-1}]
    foreach entry $pos {
        set limits [lrange $entry $llen1 $llen]
        append lines [eval getText $limits]
    }
    # nonDes means "non-destructive" ...
    set overwriteOK [expr $nonDes || ! [Bib::isBibFile]]
    getWinInfo winArray
    # Four conditions that must be met to allow over-writing ...
    if {$BibmodeVars(overwriteBuffer) && $overwriteOK && $sort && !$winArray(read-only)} {
        deleteText $beg $end
        insertText $lines
        goto $beg
        return
    }
    if {$sort} {
        set type  "Sort"
        set title "* Sort Results *"
    } else {
        set type  "Search"
        set title "* Search Results *"
    } 
    set appendResults 0
    if {$type == "Search"} {
        # This is for a search.  Do we want to append the results to a current
        # "* Search Results *" window or not?
        set searchWindows ""
        foreach window [winNames] {
            # We'll use this in a moment.
            if {[regexp "\\* Search Results \\*" [win::StripCount $window]]} {
                lappend searchWindows $window
            } 
        }
        if {$wCT != ""} {
            # The calling proc supplied a window to append to
            set appendResults $wCT
        } elseif {[llength $searchWindows]} {
            # Find out if we should append to a current search window.
            if {[askyesno "Would you like to append the search results \
              \rto the current \"Search Results\" window?"] == "yes"} {
                # Now we need to see if there's more than one.
                if {[llength $searchWindows] != 1} {
                    set appendResults [listpick -p \
                      "Please choose a window:" $searchWindows]
                } else {
                    set appendResults [lindex $searchWindows 0]
                } 
            } 
        } 
    } 
    # Create the report.
    set    t "\r$type Results for \"[win::CurrentTail]\"\r\r"
    append t "Note: Command double-click on any cite-key to return to its entry.\r"
    append t "______________________________________________________________________\r\r"
    if {$field != "" && $searchPat != "" && $entryLength != ""} {
        append t "   Search Pattern: $searchPat\r"
        append t "     Search Field: $field\r"
        append t "    Entries Found: $entryLength\r"
        append t "______________________________________________________________________\r\r"
    } 
    if {[pos::diff [minPos] [lineStart $beg]] && \
      ![regexp {\\* Search Results \\*} [win::CurrentTail]]} {
        # We only insert the "header" text if 
        # (1) there's something there, and 
        # (2) this isn't a sort of a "search results" window. 
        append t [getText [minPos] [lineStart $beg]]
    } 
    append t $lines
    append t [getText [nextLineStart $end] [maxPos]]
    # Either create a new window, or append to an existing one.
    placeBookmark
    if {$appendResults == 0} {
        set t1 "% -*-Bib-*- ([string tolower $type])\r"
        append t1 "$t\r"
        new -n $title -m "Bib" -text $t1
        set pos [minPos]
    } else {
        bringToFront $appendResults
        setWinInfo read-only 0
        goto    [maxPos]
        set pos [getPos]
        insertText "______________________________________________________________________\r"
        insertText "______________________________________________________________________\r"
        insertText $t
    }
    removeAllMarks
    Bib::MarkFile
    if {!$sort && $searchPat != ""} {
        # A little color ...
        set pos2 $pos
        if {[regexp {(all fields)|citeKey} $field]} {
	    if {![catch {search -f 1 -i 0 -s -r 1 $field $pos2} match]} {
		text::color [lindex $match 0] [lindex $match 1] 1
	    }
        } 
        while {![catch {search -f 1 -i 1 -s -r 1 $searchPat $pos2} match]} {
            text::color [lindex $match 0] [lindex $match 1] 4
            set pos2 [lindex $match 1]
        } 
    } 
    winReadOnly
    goto $pos ; insertToTop
}

proc Bib::isBibFile {{fullPath ""} {includeExample 0}} {

    set wC [win::StripCount [win::Current]]
    if  {$fullPath == ""} {set fullPath $wC} 
    set ext    [file extension $fullPath]
    set result [string match ".bib" [string tolower $ext]] 
    if {$includeExample && [regexp {Bib Mode Example} $wC]} {set result 1} 
    return $result
}

# ===========================================================================
# 
# Get the name of the field that starts before the given position, $pos. 
# The positions $top and $bottom restrict the range of the search for the
# beginning and end of the field; typically, $top and $bottom will be the
# limits of a given entry.
# 

proc Bib::getFldName {pos top} {
    set fldPat {[,  ]+([^   =,\{\}\"\']+)[  ]*=[    ]*}
    if {![catch {search -f 0 -r 1 -m 0 -i 1 -s -limit $top "$fldPat" $pos} match]} {
        set theText [eval getText $match]
        regexp -nocase $fldPat $theText allofit fldnam
        return [string tolower $fldnam]
    } else {
        return {citeKey}
    }
}

# ===========================================================================
# 
# Given the four parameters, return (or append to) a "Search Results" window
# the results of the "Search All Bib Files" search.
#

proc Bib::reportAllBibFiles {pattern field biblist results} {
    
    global BibmodeVars BibFileTailConnect
    
    if {$biblist == "-1"} {
        # Didn't find any ...
        status::msg "Sorry, couldn't find \"$pattern\"."
        return ""
    } 
    # Get the list of open windows, that we won't close later.
    set windows [winNames -f]
    # Turn off automark, because we could be opening a lot of files here.
    set oldAutoMark $BibmodeVars(autoMark)
    set BibmodeVars(autoMark) 0
    # If the field was "citeKey" or "all fields", then we know that the
    # grep search was accurate.  Otherwise, we know that we have some false
    # positives in there.  This will get ugly, but we need to open all of
    # the files and check to see if the pattern is actually in the field
    # that we need.
    set biblist2 ""
    if {$field != "citeKey" && $field != "all fields"} {
        foreach f $biblist {
            file::openQuietly $f
            if {[Bib::searchFields $field $pattern "-1"] != 0} {
                lappend biblist2 $f
            } 
            if {[lsearch $windows $f] == "-1"} {
                killWindow
            } 
        }
        status::msg ""
        if {[llength $biblist2] == 0} {
            # Didn't find any ...
            status::msg "Sorry, couldn't find \"$pattern\"."
            return ""
        } 
    } else {
        set biblist2 $biblist
    } 
    # Turn automark back on.
    set BibmodeVars(autoMark) $oldAutoMark
    if {[llength $biblist2] == 1} {
        # There was only one file.
        set title "Only one file matched \"$pattern\" :"
    } else {
        set title ""
    } 
    # Create a list for the dialog.
    set biblist3 {"List all matches"}
    foreach f $biblist2 {lappend biblist3 $BibFileTailConnect($f)} 
    # Create the results for a potential browser window.
    set     results2  "-*-Brws-*-\r\rUse the arrows keys to navigate, "
    append  results2  "\"return\" to go to a citation.\r\r"
    append  results2 $results
    # Offer the list in a listpick dialog, and add a file to the results if
    # desired.  Select one first ...
    set results3 [Bib::reportAllBibsDialog $title $field $pattern $biblist3 $results2 ""]
    set wCT      [lindex $results3 0]
    set biblist3 [concat "Cancel" [lindex $results3 1]]
    # ... And then the rest.
    while {$wCT != "-1"} {
        set title    "Choose another file, or press Cancel:"
        set results3 [Bib::reportAllBibsDialog $title $field $pattern $biblist3 $results2 $wCT]
        set wCT      [lindex $results3 0]
        set biblist3 [lindex $results3 1]
    }
}


proc Bib::reportAllBibsDialog {{title ""} field pattern biblist3 results2 {wCT ""}} {
    
    global Bib::TailFileConnect
    
    if {$title == ""} {
        set title "Multiple \"$pattern\" matches were found:"
    } 
    if {[llength $biblist3] == 2} {
        set title "List matches to open a browser window:"
    } 
    # Get the list of open windows, that we won't close later.
    set windows [winNames -f]
    set fTail [listpick -L {"List all matches"} -p $title $biblist3]
    if {$fTail == "List all matches"} {
        # If the field was "citeKey" or "all fields", then we know that the
        # grep search was accurate.  Otherwise, we know that we have some false
        # positives in there.
        if {$field != "citeKey" && $field != "all fields"} {
            dialog::alert "Note that the \"List All Matches\" option may\
              contain several false positive matches for the specific field..."
        }
        grepsToWindow "* $pattern Search Results *" $results2
        insertToTop
        # Since this can jump to any file, we stop after this.
        return "-1"
    } else {
        set f [set Bib::TailFileConnect($fTail)]
        file::openQuietly  $f
        set f2 [win::CurrentTail]
        Bib::searchFields $field $pattern $wCT
        # Current window is now the "Search Results" window.
        set wCT [win::CurrentTail]
        bringToFront $f2
        if {[lsearch $windows $fTail] == "-1"} {killWindow} 
    } 
    # ... remove it from the list ...
    set bibFSpot [lsearch  $biblist3 $fTail]
    set biblist3 [lreplace $biblist3 $bibFSpot $bibFSpot]
    if {[llength $biblist3] == 1} {
        # Only the "Cancel" remains.
        return [list "-1" ""]
    } else {
        # Return the list for another round.
        return [list $wCT $biblist3]
    } 
}

# ===========================================================================
# 
# .
